<?php

namespace App\Exports;

use Modules\Subscriptions\Models\PlanLimitation;
use Maatwebsite\Excel\Events\AfterSheet;
use PhpOffice\PhpSpreadsheet\Worksheet\PageSetup;
use PhpOffice\PhpSpreadsheet\Style\Border;
use Illuminate\Support\Facades\Auth;

class PlanLimitationExport extends BaseExport
{
    public function __construct($columns, $dateRange = [], $type = 'plan_limitation')
    {
        parent::__construct($columns, $dateRange, $type, 'Plan Limitation Report');
    }

    public function headings(): array
    {
        $modifiedHeadings = [];

        foreach ($this->columns as $column) {
            $modifiedHeadings[] = ucwords(str_replace('_', ' ', $column));
        }

        return $modifiedHeadings;
    }

    /**
     * @return \Illuminate\Support\Collection
     */
    public function collection()
    {
        $query = PlanLimitation::query();

        $query = $query->orderBy('updated_at', 'desc');

        $query = $query->get();

        $newQuery = $query->map(function ($row) {
            $selectedData = [];

            foreach ($this->columns as $column) {
                switch ($column) {

                    case 'status':
                        $selectedData[$column] = __('messages.inactive');
                        if ($row[$column]) {
                            $selectedData[$column] = __('messages.active');
                        }
                        break;

                    default:
                        $selectedData[$column] = $row[$column];
                        break;
                }
            }

            return $selectedData;
        });

        return $newQuery;
    }

    /**
     * Override registerEvents to set proper margins and column widths for Plan Limitation Report
     */
    public function registerEvents(): array
    {
        return [
            AfterSheet::class => function (AfterSheet $event) {
                $sheet = $event->sheet;
                $worksheet = $sheet->getDelegate();
                
                // Get all the parent class properties and methods we need
                $generatedBy = Auth::user()->first_name . ' ' . Auth::user()->last_name ?? 'System';
                $generatedAt = now()->format('d M Y, h:i A');
                $lastColumn = $this->getLastColumn();
                
                // Set report info (same as parent)
                $sheet->mergeCells("A1:{$lastColumn}1");
                $sheet->setCellValue('A1', $this->reportName);
                
                $currentRow = 2;
                if (!empty($this->type)) {
                    $sheet->mergeCells("A{$currentRow}:{$lastColumn}{$currentRow}");
                    $sheet->setCellValue("A{$currentRow}", 'Type: ' . ucfirst($this->type));
                    $currentRow++;
                }
                
                $sheet->mergeCells("A{$currentRow}:{$lastColumn}{$currentRow}");
                $sheet->setCellValue("A{$currentRow}", 'Generated By: ' . $generatedBy);
                $currentRow++;
                
                $sheet->mergeCells("A{$currentRow}:{$lastColumn}{$currentRow}");
                $sheet->setCellValue("A{$currentRow}", 'Generated On: ' . $generatedAt);
                
                // Apply styles (same as parent)
                $sheet->getStyle('A1')->getFont()->setBold(true)->setSize(14);
                $sheet->getStyle("A1:A{$currentRow}")->getAlignment()->setHorizontal('left');
                $headingsRow = !empty($this->type) ? 5 : 4;
                $sheet->getStyle("A{$headingsRow}:{$lastColumn}{$headingsRow}")->getFont()->setBold(true);
                
                // Remove borders from report info section (same as parent)
                $sheet->getStyle("A1:{$lastColumn}{$currentRow}")->applyFromArray([
                    'borders' => [
                        'allBorders' => [
                            'borderStyle' => Border::BORDER_NONE,
                        ],
                    ],
                ]);
                
                // Page setup - use portrait for 3 columns
                $pageSetup = $worksheet->getPageSetup();
                $pageSetup->setPaperSize(PageSetup::PAPERSIZE_A4)
                    ->setOrientation(PageSetup::ORIENTATION_PORTRAIT)
                    ->setHorizontalCentered(true);
                
                // Use fit-to-width for better display
                $pageSetup->setFitToWidth(1)
                    ->setFitToHeight(0)
                    ->setScale(100);
                
                // Set proper margins on both sides (left and right)
                $worksheet->getPageMargins()
                    ->setTop(0.5)
                    ->setBottom(0.5)
                    ->setLeft(0.5)
                    ->setRight(0.5);
                
                // Print area
                $lastRow = $worksheet->getHighestRow();
                $worksheet->getPageSetup()->setPrintArea("A1:{$lastColumn}{$lastRow}");
                
                // Wrap text for all data cells
                $sheet->getStyle("A{$headingsRow}:{$lastColumn}{$lastRow}")
                    ->getAlignment()
                    ->setWrapText(true)
                    ->setVertical('top');
                
                $worksheet->getDefaultRowDimension()->setRowHeight(-1);
                
                // Set optimized column widths for Plan Limitation export
                $lastColumnIndex = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::columnIndexFromString($lastColumn);
                $columnWidthMap = [
                    'title' => 25,
                    'description' => 60, // Wide column for long descriptions
                    'status' => 12,
                ];
                
                // Map columns to their widths
                for ($col = 1; $col <= $lastColumnIndex; $col++) {
                    $idx = $col - 1;
                    $columnKey = $this->columns[$idx] ?? '';
                    
                    if ($columnKey && array_key_exists($columnKey, $columnWidthMap)) {
                        $worksheet->getColumnDimensionByColumn($col)
                            ->setAutoSize(false)
                            ->setWidth($columnWidthMap[$columnKey]);
                    } else {
                        // Auto-size for columns not in the map
                        $worksheet->getColumnDimensionByColumn($col)->setAutoSize(true);
                    }
                }
                
                // Auto-height rows for wrapped text
                for ($row = $headingsRow; $row <= $lastRow; $row++) {
                    $worksheet->getRowDimension($row)->setRowHeight(-1);
                }
            },
        ];
    }
}
